%% Conceptual Design UAV eVTOL
clc; clear; close all;

% Dati geometrici
t_c_des=0.12;               % [-] Spessore percentuale

% Dati operativi e di missione
% h_cruise=300;             % [m] Altitudine di crociera indagine di mercato [1000 ft]
time_to=90;                 % [s] Tempo per raggiungere 1000 ft --> cos' rateo di salita = 36 km/h = 10 m/s
time_cruise=9000;           % [s] Tempo di missione voluto = 2 h e 30 min
W_pay=5;                    % [kg] Massa payload desiderata

% Dati atmosferici
rho_cruise=1.2;             % [kg/m^3] Densità dell'aria a h=1000 ft
R=287;                      % [J/(kg K)] Costante universale dei gas
y=1.4;                      % [-] Rapporto calori specifici
T=286.17;                   % [K] Temperatura alla quota h=1000 ft
a_cruise=sqrt(y*R*T);       % [m/s] Velocità del suono 
rhosl=1.225;                % [kg/m^3] Densità dell'aria al livello del mare
g=9.81;                     % [m/s^2] Accelerazione gravitazionale

% Dati strutturali e prestazioni
MTOW_iniz=25;               % [kg] Stima iniziale peso massimo al decollo 
e=0.8;                      % [-] Fattore di Oswald

% Dati batteria semi-solida selezionata
n_batt=2;                   % [-] Numero batterie
C=22;                       % [Ah] Capacità
Volt=44.4;                  % [V] Tensione nominale
BED=320;                    % [Wh/kg] Battery energy density
DoD=0.85;                   % [-] Efficienza di scarica
W_batt=3.68*2;              % [kg] Peso dell'intero pacco batteria

% Dati eliche VTOL T-Motor selezionate
n_VTOL=4;                               % [-] Numero eliche VTOL
r_VTOL=0.559/2;                         % [m] Raggio singola elica
W_eliche_VTOL=4*0.096;                  % [kg] Massa totale eliche VTOL

% Dati motori VTOL T-Motor selezionati
W_mot_VTOL=0.345*n_VTOL;                % [kg] Massa totale motori VTOL
W_ESC_VTOL=4*0.0735;                    % [kg] Massa dei 4 ESC selezionati

% Dati elica crociera T-Motor selezionata
W_elica_cruise=2*0.0218;                  % [kg] Massa elica cruise

% Dati motore cruise T-Motor selezionato
W_mot_cruise=2*0.304;                     % [kg] Massa motore
W_ESC_cruise=2*0.182;                     % [kg] Massa ESC selezionato

% Massa flight control computer
Wfc=1;      % [kg]

% Fattori di conversione
kg2lb=2.2046;               % Conversione kg to lb
lb2kg=0.4536;               % Conversione lb to kg
m2ft=3.28084;               % Conversione m to feet
w2hp=0.00134;               % Conversione watt to horse-power 
               
% Design variables 
S_vett=[1.3 1.4 1.5 1.6];       % [m^2] Superficie alare
b_vett=[3 3.2 3.5 3.8 4 4.2];   % [m] Apertura alare
lambda_vett=[0.4 0.5 0.6];  % [-] Taper ratio moderato
v_vett=[20 25 30];          % [m/s] Velocità di crociera

indice_contatore=0;         % [-] Contatore per costruire matrice di configurazione

% ciclo di dimensionamento
for i = 1:length(S_vett)
    for j=1:length(b_vett)
        for u=1:length(lambda_vett)
            for n=1:length(v_vett)

                % Dichiarare variabili che si aggiornano
                S_des = S_vett(i);
                b_des = b_vett(j);
                lambda = lambda_vett(u);
                v_cruise=v_vett(n);
                
                delta_WTO=1;               % inizializzazione delta per entrare nel ciclo
                toll=0.1;                  % tolleranza
                WTO_curr=MTOW_iniz;        % s'inizializza il primo peso corrente con la stima iniziale di MTOW            
                
                while abs(delta_WTO)>toll

                    % WING GEOMETRY               
                    W_S_ref=WTO_curr/(S_des);           % carico alare [kg/m^2]
                    AR_ref=b_des^2/S_des;               % allungamento alare [m]
                    c_root=2*S_des/(b_des*(1+lambda));  % corda alla radice [m]
                    c_tip=lambda*c_root;                % corda al tip [m]
                    c_media=(2/3)*c_root*(1+lambda+lambda^2)/(1+lambda);    % corda media aerodinamica [m]                    
                    % ipotizzando ala in cui la corda al tip è centrata rispetto alla corda alla radice
                    Gamma_le=atan((c_root-c_tip)/b_des);                % angolo di freccia al bordo d'attacco [rad]
                    Gamma_25=atan((c_root-c_tip)*(0.5-0.25)/(b_des/2)); % angolo di freccia al 25% della corda [rad]
                    Gamma_te=atan((c_root-c_tip)*(0.5-1)/(b_des/2));    % angolo di freccia al bordo di fuga [rad]

                    % AERODINAMICA
                    CL_des=2*W_S_ref*g/(rho_cruise*v_cruise^2);  % Coefficiente di portanza [-]
                    Cd0=0.050;                                   % Coeff. resistenza parassita [-] da indagine di mercato
                    k=1/(pi*e*AR_ref);
                    Cdi=k*CL_des^2;                             % Coeff. resistenza indotta [-]
                    Cd=Cd0+Cdi;                                 % Coeff. resistenza totale [-]
                    E=CL_des/Cd;                                % Efficienza [-]

                    % CALCOLO POTENZE
                    % Potenza cruise
                    eta_cruise=0.85;                   % efficienza propulsiva
                    % P/W = T*v/W = D*v/W per cui D è somma di contributo parassita + contributo indotto
                    P_W_par = 0.5 * rho_cruise * v_cruise^3 * Cd0 ./ W_S_ref;     % parassita [W/kg]
                    P_W_ind = 0.5 * rho_cruise * v_cruise^3 * Cdi ./ W_S_ref;     % indotta  [W/kg]
                    P_W_cruise = (P_W_par + P_W_ind) / eta_cruise;   % [W/kg]
                    P_W_cruise=P_W_cruise/(rho_cruise/rhosl)^0.75;                % scalo sulla densità in quota
                    P_curr_cruise = P_W_cruise * WTO_curr;                        % Potenza richiesta in crociera [W]
                    T_eng_cruise=P_curr_cruise/(v_cruise*g);                      % Spinta in crociera [kg]

                    % Potenza HOVER (utile solo per calcolare Take-off e Landing)
                    A_prop_VTOL=n_VTOL*pi*r_VTOL^2;         % Area totale eliche [m^2]
                    V_i=sqrt((WTO_curr*g)/(2*rho_cruise*A_prop_VTOL));  % Velocità indotta [m/s]
                    P_hover=(WTO_curr*g)*V_i;               % Potenza richiesta in hover dove T=W

                    % Potenza Take-off
                    P_to=P_hover*1.2;   % fattore che determina dispendio energetico maggiore rispetto a hover
                    T_to=P_to/(V_i*g);    % spinta di ogni singola elica [kg]
        
                    % Potenza Landing
                    P_la=P_hover*0.6;   % fattore che determina dispendio energetico minore rispetto a hover

                    % Potenza totale lungo il profilo di missione (take-off + cruise + landing)
                    P_tot=P_to*(time_to/(time_to+time_cruise+time_to))+P_curr_cruise*(time_cruise/(time_to+time_cruise+time_to))+P_la*(time_to/(time_to+time_cruise+time_to));
                    
                    % AUTONOMIA ORARIA garantita dalle due batterie scelte
                    endurance=n_batt*(C*DoD)/(P_tot/Volt);         % [h]
                    range=(v_cruise*3.6)*endurance;                % [km]

                    % PESI                
                    
                    % Massa ala
                    SF=1.5;             % safety factor
                    n_lim=3.8;          % limit load factor preso dalla JAR-23 (https://www.fzt.haw-hamburg.de/pers/Scholz/HOOU/AircraftDesign_10_Mass.pdf)
                    n_ult=n_lim*SF;     % ultimate load factor
                    Iw=(n_ult*AR_ref.^1.5.*((WTO_curr-W_batt)/WTO_curr).^0.5.*(1+2*lambda).*(W_S_ref/0.4536)*0.3048^2.*(S_des*m2ft^2).^1.5.*1e-6)./((t_c_des).*(cos(Gamma_25)).^2.*(1+lambda));  % indice della geometria dell'ala
                    W_wing=(4.24.*Iw+0.57.*(S_des*m2ft^2))*lb2kg;     % massa ala [kg]

                    % Massa fusoliera    
                    hf_max=0.28;            % Altezza massima della fusoliera da CAD [m]
                    wf_max=0.3;             % Larghezza massima della fusoliera da CAD [m]
                    Lf=2;                   % Lunghezza della fusoliera [m]
                    DF=(wf_max+hf_max)./2;
                    W_fus=(1.35.*((Lf.*DF)*m2ft).^1.28)*lb2kg;      % massa fusoliera [kg]

                    % Weight tail
                    W_tail=0.07*WTO_curr;   %  [kg] coda calcolata come una frazione dell'MTOW
                             
                    % Peso a vuoto               
                    Weo=W_fus+W_wing+W_tail+W_batt+W_eliche_VTOL+W_mot_VTOL+W_ESC_VTOL+W_elica_cruise+W_ESC_cruise+W_mot_cruise+Wfc;
                
                    WTO_precedente=WTO_curr;
                
                    WTO_curr=Weo+W_pay;
                
                    delta_WTO=WTO_curr-WTO_precedente;
                end
                
                indice_contatore=indice_contatore+1;
                                       
                % Memorizzazione dei dati nella matrice di memoria
                              
                % Geometry
                W_S_des_memo(indice_contatore) = W_S_ref;
                S_ref_memo(indice_contatore) = S_des;
                b_ref_memo(indice_contatore) = b_des;
                AR_memo(indice_contatore)=AR_ref;
                lambda_memo(indice_contatore)=lambda;
                Gamma_memo_25(indice_contatore)=Gamma_25;
                
                % Performance
                P_memo_tot_richiesta(indice_contatore)=P_tot;
                Endurance_memo(indice_contatore)=endurance;
                T_eng_memo(indice_contatore)=T_eng_cruise;  
                T_to_memo(indice_contatore)=T_to;
                Cl_memo(indice_contatore)=CL_des;
                Cd_memo(indice_contatore)=Cd;
                E_memo(indice_contatore)=E;
                v_memo(indice_contatore)=v_cruise;
                range_memo(indice_contatore)=range;
                                                                
                % Weights
                MTOW_memo(indice_contatore) = WTO_curr;
                W_empty_memo(indice_contatore) = Weo;
                W_battery_memo(indice_contatore) = W_batt;

                %Potenze
                P_W_cruise_memo(indice_contatore) = P_W_cruise;
                P_W_to(indice_contatore) = P_to/WTO_curr;
                P_W_landing(indice_contatore) = P_la/WTO_curr;

            end
        end
    end                                     
end

W_pay=W_pay*ones(size(MTOW_memo));

% Creazione della matrice di configurazione con trasposizione
config_matrix = [S_ref_memo;
    b_ref_memo;
    AR_memo;
    lambda_memo;
    Gamma_memo_25*(180/pi);
    W_S_des_memo;
    MTOW_memo;
    W_empty_memo;
    W_battery_memo;
    W_pay;
    P_memo_tot_richiesta;
    T_eng_memo;
    T_to_memo;
    v_memo;
    Endurance_memo;
    range_memo;
    Cl_memo;
    Cd_memo;
    E_memo;
    P_W_cruise_memo;
    P_W_to;
    P_W_landing
    ]';

% LIMITE SULL'MTOW
Nconfigurazioni=size(config_matrix,1);
for l=Nconfigurazioni:-1:1
    if config_matrix(l,7)>25
        config_matrix(l,:)=[];
        Nconfigurazioni=Nconfigurazioni-1;
    end
end

% Limite su Endurance
Nconfigurazioni=size(config_matrix,1);
for l=Nconfigurazioni:-1:1
    if (config_matrix(l,15)*60)<150
        config_matrix(l,:)=[];
        Nconfigurazioni=Nconfigurazioni-1;
    end
end

% Limite su Cl
Nconfigurazioni=size(config_matrix,1);
for l=Nconfigurazioni:-1:1
    if config_matrix(l,17)>0.7
        config_matrix(l,:)=[];
        Nconfigurazioni=Nconfigurazioni-1;
    end
end

config_top=config_matrix(:,1:end);
% Definisco i nomi delle colonne
titoli_colonne = {'S [m^2]','b [m]','AR','λ','Λ_25 [°]','W/S [kg/m^2]','MTOW [kg]','OEW [kg]','W_batt [kg]','W_pay [kg]','P_tot [W]','T_cruise [kg]','T_T/O [kg]','V_cruise [m/s]','Endurance [h]','Range [km]','C_{l}','C_{d}','E','P_cruise','P_to','P_land'};
% Converto la matrice in una tabella e assegno i titoli
tabella = array2table(config_top, 'VariableNames', titoli_colonne);

max_endurance=max(config_matrix(:,15));
rowIndex = find(config_matrix(:,15) == max_endurance);
Config_vincente=config_matrix(rowIndex,:);
UAV=array2table(Config_vincente,'VariableNames',titoli_colonne);

%% Payload range diagram  
W_pay=5;
payload=linspace(0,W_pay,11);                               % Massa di payload che varia tra 0 e quella massima [kg]
W_ref=Config_vincente(1,7);                                 % MTOW con massimo payload [kg]
MTOW=Config_vincente(1,8)+payload;                          % MTOW che varia con il payload [kg]
R_pay_ref=Config_vincente(1,16);  % Range a payload massimo [km], ossia il range minimo
R_pay=R_pay_ref*(W_ref./MTOW);                              % Range con payload che varia [km]
figure(1)
% Linea principale rossa
plot(R_pay, MTOW, 'LineWidth', 1.1, 'Color', 'b');
xlabel('Range [km]')
ylabel('MTOW [kg]')
title('Payload Range Diagram');
xlim([R_pay_ref R_pay(1)]);
ylim([MTOW(1) W_ref]);
hold on  % meglio di hold all

% Pallino vuoto per W_pay_max
plot(R_pay_ref, W_ref, 'o', 'Color', 'r', 'MarkerSize', 6, 'LineWidth', 1.0)
text(R_pay_ref-1, W_ref, 'W_{pay_{max}}', 'HorizontalAlignment', 'right', ...
    'VerticalAlignment', 'bottom', 'FontSize', 11, 'Color', 'k')

% Pallino vuoto per W_pay=0
plot(R_pay_ref, Config_vincente(1,8), 'o', 'Color', 'r', 'MarkerSize', 6, 'LineWidth', 1.0)
text(R_pay_ref-1, Config_vincente(1,8), 'W_{pay}=0', 'HorizontalAlignment', 'right', ...
    'VerticalAlignment', 'middle', 'FontSize', 11, 'Color', 'k')

grid on

%%
decollo=linspace(0,90,100);
cruise=linspace(90,9000,100000);
atterraggio=linspace(9000,9090,100);
transizione1=90;
transizione2=9000;

potenza_cruise=(Config_vincente(1,20)*Config_vincente(1,7))*ones(1,length(cruise));
potenza_decollo=(Config_vincente(1,21)*Config_vincente(1,7))*ones(1,length(decollo));
potenza_atterraggio=(Config_vincente(1,22)*Config_vincente(1,7))*ones(1,length(atterraggio));
transition1=linspace(Config_vincente(1,21)*Config_vincente(1,7),Config_vincente(1,20)*Config_vincente(1,7));
transition2=linspace(Config_vincente(1,22)*Config_vincente(1,7),Config_vincente(1,20)*Config_vincente(1,7));
x_vert1 = transizione1 * ones(size(transition1));
x_vert2 = transizione2 * ones(size(transition2));

figure(2)
plot(decollo,potenza_decollo,'LineWidth',1.5,'Color','b');  % blu scuro
hold on
plot(cruise,potenza_cruise,'LineWidth',1.2,'Color','k'); % grigio scuro
plot(atterraggio,potenza_atterraggio,'LineWidth',1.5,'Color','r'); % verde scuro

% Rette verticali di transizione
plot(x_vert1,transition1,'LineWidth',1.4,'Color',[0.8 0 0.8],'LineStyle','--'); % ciano
plot(x_vert2,transition2,'LineWidth',1.4,'Color',[0 0.7 0.7],'LineStyle','--'); % magenta chiaro

xlim([0,9090]);
ylim([0,3000]);
xlabel('Time [s]');
ylabel('Power [w]')
grid on

legend('Take-off','Cruise','Landing','Transition T/O --> Cruise','Transition Cruise --> Landing')